<?php
if (!defined('ABSPATH')) exit;

add_action('wp_enqueue_scripts', function() {
    if (is_admin()) return;
    wp_enqueue_style('warforms-reboot-frontend', WARFORMS_REBOOT_URL . 'assets/frontend.css', [], WARFORMS_REBOOT_VERSION);
    wp_enqueue_script('warforms-reboot-frontend', WARFORMS_REBOOT_URL . 'assets/frontend.js', [], WARFORMS_REBOOT_VERSION, true);
});



/**
 * Render a single field HTML block.
 *
 * Layout rendering uses this so we can place fields into rows/columns.
 */
function warforms_reboot_render_field_html(array $field, array $values, array $errors): string {
    $type = $field['type'] ?? 'text';
    $name = $field['name'] ?? '';
    $label = $field['label'] ?? '';
    $required = !empty($field['required']);
    if (!$name || !$label) return '';

    $field_key = 'wf[' . $name . ']';
    $val = $values[$name] ?? null;
    $err = $errors[$name] ?? '';

    // Normalize options (arrays or newline strings)
    $options = [];
    if (isset($field['options'])) {
        if (is_array($field['options'])) {
            $options = $field['options'];
        } elseif (is_string($field['options'])) {
            $lines = preg_split("/\r?\n/", $field['options']);
            $options = array_values(array_filter(array_map('trim', $lines), function($s){ return $s !== ''; }));
        }
    }

    ob_start();

    echo '<div class="wf-field wf-type-' . esc_attr($type) . '">';
    $for_id = ($type === 'phone') ? ('wf_' . $name . '_area') : ('wf_' . $name);
    echo '<label for="' . esc_attr($for_id) . '">' . esc_html($label);
    if ($required) echo ' <span aria-hidden="true">*</span>';
    echo '</label>';

    $aria_req = $required ? ' aria-required="true"' : '';
    $req_attr = $required ? ' required' : '';

    if ($type === 'textarea') {
        echo '<textarea id="wf_' . esc_attr($name) . '" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . ' rows="4">' . esc_textarea(is_string($val) ? $val : '') . '</textarea>';
    } elseif ($type === 'email') {
        echo '<input id="wf_' . esc_attr($name) . '" type="email" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . ' value="' . esc_attr(is_string($val) ? $val : '') . '">';
    } elseif ($type === 'url') {
        echo '<input id="wf_' . esc_attr($name) . '" type="url" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . ' value="' . esc_attr(is_string($val) ? $val : '') . '">';
    } elseif ($type === 'date') {
        echo '<input id="wf_' . esc_attr($name) . '" type="date" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . ' value="' . esc_attr(is_string($val) ? $val : '') . '">';
    } elseif ($type === 'phone') {
        // Render a NA phone input as (###) ###-####, store digits in hidden wf[name].
        $digits = is_string($val) ? preg_replace('/\D+/', '', $val) : '';
        if (strlen($digits) === 11 && $digits[0] === '1') {
            $digits = substr($digits, 1);
        } elseif (strlen($digits) > 11) {
            $digits = substr($digits, -10);
        }

        $area  = substr($digits, 0, 3);
        $pref  = substr($digits, 3, 3);
        $line  = substr($digits, 6, 4);

        echo '<div class="wf-phone" data-wf-phone="wrap">';
        echo '<span class="wf-phone-prefix" aria-hidden="true">+1</span>';
        echo '<span class="wf-phone-sep" aria-hidden="true">(</span>';
        echo '<input id="wf_' . esc_attr($name) . '_area" class="wf-phone-part" data-wf-phone="area" inputmode="numeric" pattern="\d{3}" placeholder="555" name="wf_phone[' . esc_attr($name) . '][area]" value="' . esc_attr($area) . '">';
        echo '<span class="wf-phone-sep" aria-hidden="true">)</span>';
        echo '<input class="wf-phone-part" data-wf-phone="prefix" inputmode="numeric" pattern="\d{3}" placeholder="555" name="wf_phone[' . esc_attr($name) . '][prefix]" value="' . esc_attr($pref) . '">';
        echo '<span class="wf-phone-sep" aria-hidden="true">-</span>';
        echo '<input class="wf-phone-part" data-wf-phone="line" inputmode="numeric" pattern="\d{4}" placeholder="5555" name="wf_phone[' . esc_attr($name) . '][line]" value="' . esc_attr($line) . '">';

        // Hidden canonical field that validation expects.
        echo '<input id="wf_' . esc_attr($name) . '" type="hidden" name="' . esc_attr($field_key) . '" value="' . esc_attr($digits) . '">';
        echo '</div>';
    } elseif (in_array($type, ['select','radio'], true)) {
        $current = is_string($val) ? $val : '';
        if ($type === 'select') {
            echo '<select id="wf_' . esc_attr($name) . '" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . '>';
            echo '<option value="">— Select —</option>';
            foreach ($options as $opt) {
                $opt = (string)$opt;
                $sel = selected($current, $opt, false);
                echo '<option value="' . esc_attr($opt) . '"' . $sel . '>' . esc_html($opt) . '</option>';
            }
            echo '</select>';
        } else { // radio
            echo '<div class="wf-choice-group">';
            foreach ($options as $opt) {
                $opt = (string)$opt;
                $checked = checked($current, $opt, false);
                echo '<label><input type="radio" name="' . esc_attr($field_key) . '" value="' . esc_attr($opt) . '"' . $checked . ($required ? ' required' : '') . '> ' . esc_html($opt) . '</label>';
            }
            echo '</div>';
        }
    } elseif (in_array($type, ['multiselect','checkbox_group'], true)) {
        $current = is_array($val) ? $val : [];
        if ($type === 'multiselect') {
            echo '<select id="wf_' . esc_attr($name) . '" name="' . esc_attr($field_key) . '[]" multiple size="' . esc_attr(min(6, max(3, count($options)))) . '"' . ($required ? ' required' : '') . '>';
            foreach ($options as $opt) {
                $opt = (string)$opt;
                $sel = in_array($opt, $current, true) ? ' selected' : '';
                echo '<option value="' . esc_attr($opt) . '"' . $sel . '>' . esc_html($opt) . '</option>';
            }
            echo '</select>';
        } else { // checkbox_group
            echo '<div class="wf-choice-group">';
            foreach ($options as $opt) {
                $opt = (string)$opt;
                $checked = in_array($opt, $current, true) ? ' checked' : '';
                echo '<label><input type="checkbox" name="' . esc_attr($field_key) . '[]" value="' . esc_attr($opt) . '"' . $checked . '> ' . esc_html($opt) . '</label>';
            }
            echo '</div>';
        }
    } elseif ($type === 'currency') {
        $allowed = is_array($field['allowed'] ?? null) ? $field['allowed'] : ['USD'];
        $current = is_array($val) ? $val : ['amount' => is_string($val) ? $val : '', 'currency' => $allowed[0] ?? 'USD'];
        $amount = is_array($current) ? ($current['amount'] ?? '') : '';
        $curr = is_array($current) ? ($current['currency'] ?? ($allowed[0] ?? 'USD')) : ($allowed[0] ?? 'USD');
        echo '<div class="wf-currency">';
        echo '<input type="text" inputmode="decimal" placeholder="0.00" name="wf_currency[' . esc_attr($name) . '][amount]" value="' . esc_attr((string)$amount) . '"' . ($required ? ' required' : '') . '>';
        echo '<select name="wf_currency[' . esc_attr($name) . '][currency]">';
        foreach ($allowed as $c) {
            $c = (string)$c;
            $sel = selected($curr, $c, false);
            echo '<option value="' . esc_attr($c) . '"' . $sel . '>' . esc_html($c) . '</option>';
        }
        echo '</select>';
        echo '</div>';
    } else {
        echo '<input id="wf_' . esc_attr($name) . '" type="text" name="' . esc_attr($field_key) . '"' . $req_attr . $aria_req . ' value="' . esc_attr(is_string($val) ? $val : '') . '">';
    }

    if ($err) {
        echo '<div class="wf-error">' . esc_html($err) . '</div>';
    }
    echo '</div>';

    return ob_get_clean();
}

add_shortcode('warform', function($atts) {
    $atts = shortcode_atts(['id' => 0], $atts, 'warform');
    $form_id = (int)$atts['id'];
    if ($form_id <= 0) return '<!-- warform: missing id -->';

    $post = get_post($form_id);
    if (!$post || $post->post_type !== 'warform') return '<!-- warform: invalid id -->';

    $schema = warforms_reboot_get_schema($form_id);

    $token = isset($_GET['wf_token']) ? sanitize_text_field(wp_unslash($_GET['wf_token'])) : '';
    $flash = null;
    if ($token) {
        $flash = get_transient('warforms_reboot_flash_' . $token);
        if (is_array($flash) && (int)($flash['form_id'] ?? 0) === $form_id) {
            // one-time
            delete_transient('warforms_reboot_flash_' . $token);
        } else {
            $flash = null;
        }
    }

    $values = is_array($flash['values'] ?? null) ? $flash['values'] : [];
    $errors = is_array($flash['errors'] ?? null) ? $flash['errors'] : [];
    $status = $flash['status'] ?? '';

    ob_start();
    echo '<form class="warforms-reboot-form" method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';

    // Notices
    if ($status === 'success') {
    // Step 11: dynamic success notice (schema confirmation message if present).
    $msg = 'Thanks! Your submission was received.';
    $conf = isset($schema['confirmation']) && is_array($schema['confirmation']) ? $schema['confirmation'] : [];
    $conf_type = isset($conf['type']) ? (string) $conf['type'] : '';
    if ($conf_type === 'message') {
        $custom = isset($conf['message']) ? (string) $conf['message'] : '';
        if (trim($custom) !== '') {
            $msg = $custom;
        }
    }
    echo '<div class="wf-notice wf-success">' . wp_kses_post(wpautop($msg)) . '</div>';
} elseif ($status === 'fail') {

        echo '<div class="wf-notice wf-fail">Please fix the errors below and try again.</div>';
        if (!empty($errors['_form'])) {
            echo '<div class="wf-notice wf-fail wf-form-error">' . esc_html((string)$errors['_form']) . '</div>';
        }
    }

    // core hidden fields
    echo '<input type="hidden" name="action" value="warforms_submit">';
    echo '<input type="hidden" name="wf_form_id" value="' . (int)$form_id . '">';
    wp_nonce_field('warforms_submit_' . $form_id, 'wf_nonce');

    // Honeypot (must never show)
    echo '<div class="wf-hp" aria-hidden="true" tabindex="-1" style="display:none !important; position:absolute !important; left:-9999px !important; top:-9999px !important;">';
    echo '<label>Leave this field empty<input type="text" name="wf_hp" value=""></label>';
    echo '</div>';

    if (empty($schema['fields'])) {
        echo '<p><em>This form has no fields yet.</em></p>';
    } else {
        // Build a lookup by field id for layout rendering.
        $fields_by_id = [];
        foreach ($schema['fields'] as $f) {
            if (is_array($f) && !empty($f['id'])) {
                $fields_by_id[(string)$f['id']] = $f;
            }
        }

        $layout = isset($schema['layout']) && is_array($schema['layout']) ? $schema['layout'] : null;

        if ($layout && !empty($layout['rows']) && is_array($layout['rows'])) {
            $seen = [];

            foreach ($layout['rows'] as $row) {
                if (!is_array($row)) continue;
                $cols = isset($row['cols']) ? max(1, min(3, (int)$row['cols'])) : 1;
                $cols_data = isset($row['columns']) && is_array($row['columns']) ? $row['columns'] : [];
                echo '<div class="wf-row wf-cols-' . (int)$cols . '">';

                for ($c = 0; $c < $cols; $c++) {
                    echo '<div class="wf-col">';
                    $ids = isset($cols_data[$c]) && is_array($cols_data[$c]) ? $cols_data[$c] : [];
                    foreach ($ids as $fid) {
                        $fid = (string)$fid;
                        if ($fid === '' || isset($seen[$fid]) || empty($fields_by_id[$fid])) continue;
                        $seen[$fid] = true;
                        echo warforms_reboot_render_field_html($fields_by_id[$fid], $values, $errors);
                    }
                    echo '</div>';
                }

                echo '</div>';
            }

            // Unassigned (or any remaining) fields render after layout.
            $unassigned = isset($layout['unassigned']) && is_array($layout['unassigned']) ? $layout['unassigned'] : [];
            foreach ($unassigned as $fid) {
                $fid = (string)$fid;
                if ($fid === '' || isset($seen[$fid]) || empty($fields_by_id[$fid])) continue;
                $seen[$fid] = true;
                echo warforms_reboot_render_field_html($fields_by_id[$fid], $values, $errors);
            }

            // Any field not referenced anywhere still renders, in schema order.
            foreach ($schema['fields'] as $f) {
                if (!is_array($f) || empty($f['id'])) continue;
                $fid = (string)$f['id'];
                if ($fid === '' || isset($seen[$fid])) continue;
                $seen[$fid] = true;
                echo warforms_reboot_render_field_html($f, $values, $errors);
            }
        } else {
            // Default: render in schema order.
            foreach ($schema['fields'] as $field) {
                if (!is_array($field)) continue;
                echo warforms_reboot_render_field_html($field, $values, $errors);
            }
        }
    }
echo '<p><button type="submit">Submit</button></p>';
    echo '</form>';

    return ob_get_clean();
});
